--------------------------------------------------------------------------
-- This prefab file is for loading autogenerated NPC prefabs
--------------------------------------------------------------------------

local Npc = require "components.npc"
local NpcAutogenData = require "prefabs.npc_autogen_data"
local fmodtable = require "defs.sound.fmodtable"
local prefabutil = require "prefabs.prefabutil"

---------------------------------------------------------------------------------------


local function OnActivate(inst, player)
	assert(not TheDungeon:GetDungeonMap():IsDebugMap())

	inst.components.conversation:ActivatePrompt(player)
end

local function OnDeactivate(inst, player)
	assert(not TheDungeon:GetDungeonMap():IsDebugMap())
	inst.components.conversation:TryDeactivatePrompt(player)
end

local function CanInteract(inst, player, is_focused)
	if is_focused then
		return true
	elseif player.components.playercontroller:IsPlacingOrRemoving() then
		return false, "placing or removing prop"
	elseif inst:HasTag("INLIMBO") then
		return false, "in limbo"
	elseif TheDungeon:GetDungeonMap():IsDebugMap() then
		return false, "debug map"
	elseif inst.components.interactable.force_disable then
		return false, "force_disable"
	end
	return inst.components.conversation:CanStartModalConversation(player)
end

local function PerformInteract(inst, player)
	local prompt_target = TheDungeon.HUD:GetPromptTarget()
	local convo_target = inst.components.conversation:GetTarget()
	if not prompt_target or not convo_target then
		-- Prompt was cleared, but we're trying to continue activation so
		-- restart it.
		-- TODO(convo): Re-evaluate this decision along with always calling
		-- DeactivatePrompt in Conversation:_EndConversation.
		OnActivate(inst, player)
	end
	prompt_target = TheDungeon.HUD:GetPromptTarget()
	convo_target = inst.components.conversation:GetTarget()
	if prompt_target ~= inst then
		-- If the prompt target has changed away from us, then the player isn't
		-- trying to interact with us. Probably another player started another
		-- interaction.
		TheLog.ch.Convo:printf("Npc prompt target changed from us (%s) to '%s'. Cancelling interaction.", inst, prompt_target)
		dbassert(#AllPlayers > 1, "How did prompt switch focus without another player?")
		return false
	end

	assert(prompt_target == inst, prompt_target)
	assert(convo_target == player, convo_target)

	-- Before this point, the npc had attract text above them but the player
	-- wasn't locked into the convo.

	local success = inst.components.conversation:BeginModalConversation()
	if success then
		-- Clear us from interactable so we never rely on the interaction target.
		player.components.playercontroller:SetInteractTarget(nil)
	end
end

---------------------------------------------------------------------------------------

local function OnHomeChanged(inst, home)

	local npc = inst.components.npc
	local role = npc.role

	assert(inst:IsValid())
	-- don't fire off registernpc events during TransitionLevel world destruction
	-- otherwise invalid references will be held by TheDungeon's castmanager
	if not TheWorld.is_destroying then
		TheWorld:PushEvent("registernpc", { npc = inst, role = role })
	end
end

---------------------------------------------------------------------------------------

local function CreateMouth(prefabname, bank, build)
	local inst = CreateEntity()
	inst:SetPrefabName(prefabname)

	inst.entity:AddTransform()
	inst.entity:AddAnimState()
	inst.entity:AddFollower()

	inst:AddTag("YESCLICK")
	inst:AddTag("FX")  -- return body when doing mouse over checks
	inst.persists = false

	inst.Transform:SetTwoFaced()

	inst.AnimState:SetBank(bank)
	inst.AnimState:SetBuild(build)
	inst.AnimState:PlayAnimation("neutral_mouth_idle")

	return inst
end

local function CreateHead(bank, build)
	local inst = CreateEntity()

	inst.entity:AddTransform()
	inst.entity:AddAnimState()
	inst.entity:AddFollower()

	inst:AddTag("YESCLICK")
	inst:AddTag("FX")
	inst.persists = false

	inst.Transform:SetTwoFaced()

	inst.AnimState:SetBank(bank)
	inst.AnimState:SetBuild(build)
	inst.AnimState:PlayAnimation("idle")

	return inst
end

local function OnSave(inst, data)
	data.state = inst.state
end

local function OnLoad(inst, data)
	inst.state = data ~= nil and data.state or nil
end

function MakeAutogenMouth(name, params, is_debug)
	local assets =
	{
		Asset("ANIM", "anim/npc_template_mouth.zip"),
	}
	local prefabs = {}

	local head_build = params.head or ("%s_head"):format(name)

	local mouth_bank = "npc_template_mouth" -- each npc shares the same mouth animations
	local mouth_build = head_build -- mouth symbols are pulled from the head build

	local function fn(prefabname)
		local inst = CreateMouth(prefabname, mouth_bank, mouth_build)
		return inst
	end

	return Prefab(name.."_mouth", fn, assets, prefabs, nil, NetworkType_None)
end

function MakeAutogenNpc(name, params, is_debug)
	local assets =
	{
		Asset("PKGREF", "scripts/prefabs/autogen/npc/".. name ..".lua"),
		Asset("PKGREF", "scripts/prefabs/npc_autogen.lua"),
		Asset("PKGREF", "scripts/prefabs/npc_autogen_data.lua"),
		Asset("ANIM", "anim/npc_template.zip"),
		Asset("ANIM", "anim/npc_template_mouth.zip"),
		Asset("ANIM", "anim/npc_character_specific.zip"),
	}

	local prefabs = {
		"npcmarker"
	}

	local body_build = params.build or name
	local body_bank = params.bank or "npc_template"

	-- NPCs have unique head animations because each head is shaped so differently.
	local head_build = params.head or ("%s_head"):format(name)
	local head_bank = head_build

	local mouth_bank = "npc_template_mouth" -- each npc shares the same mouth animations
	local mouth_build = head_build -- mouth symbols are pulled from the head build

	prefabutil.CollectAssetsForAnim(assets, body_build, nil, params.bankfile, debug)
	prefabutil.CollectAssetsForAnim(assets, head_build, nil, params.bankfile, debug)
	prefabutil.CollectAssetsAndPrefabsForScript(assets, prefabs, name, params.script, params.script_args, debug)

	local placer_prefab = nil
	if params.home then
		placer_prefab = params.home .."_placer"
		table.insert(prefabs, placer_prefab)
	end

	local function fn(prefabname)
		local inst = CreateEntity()
		inst:SetPrefabName(prefabname)

		inst.tuning = TUNING.npc.generic

		inst.entity:AddTransform()
		inst.entity:AddAnimState()

		inst.entity:AddSoundEmitter()

		MakeNpcPhysics(inst, 1.8) --bigger than usual to prevent player's small collision size from getting right up in their grill + so they don't walk into tiny crevices

		inst:AddTag("npc")
		inst:AddTag("character")

		inst.Transform:SetTwoFaced()
		-- copy the initial rotation/facing, then let npc facing directions be desynced based on local interactions
		inst:DoTaskInTicks(0, function()
			inst.Transform:SetNetworkHistoryIgnoreFlags(NETHISTORYIGNOREFLAG_ROTATION)
		end)
	
		inst.AnimState:SetBank(body_bank)
		inst.AnimState:SetBuild(body_build)
		inst.AnimState:PlayAnimation("idle", true)
		inst.AnimState:SetShadowEnabled(true)

		inst.AnimState:SetRimEnabled(true)
		inst.AnimState:SetRimSize(1.5)
		inst.AnimState:SetRimSteps(2)

		inst.AnimState:SetSilhouetteColor(0/255, 0/255, 0/255, 0.2)
		inst.AnimState:SetSilhouetteMode(SilhouetteMode.Have)

		inst.head = CreateHead(head_bank, head_build)
		inst.head.entity:SetParent(inst.entity)
		inst.head.Follower:FollowSymbol(inst.GUID, "head_follow")

		-- spawn our mouth - need to defer it to start to next frame as a prefab can't spawn a prefab from its constructor
		inst:DoTaskInTime(0, function()
			inst.mouth = SpawnPrefab(name.."_mouth")
			inst.mouth.entity:SetParent(inst.head.entity)
			inst.mouth.Follower:FollowSymbol(inst.head.GUID, "mouth_swap")
			inst.mouth:ListenForEvent("sfx-speech_blah", function() inst:PushEvent("speech_blah") end)
			inst:PushEvent("mouthacquired")
		end)

		if params.held_item then
			inst.AnimState:HideLayer("ARM_normal")
			inst.AnimState:ShowLayer("ARM_carry")
		else
			inst.AnimState:HideLayer("ARM_carry")
			inst.AnimState:ShowLayer("ARM_normal")
		end

		inst:AddComponent("locomotor")
		inst.components.locomotor:SetWalkSpeed(1.85)

		inst:AddComponent("interactable")
			:SetRadius(3.5)
			:SetInteractionOffset(Vector3.unit_x * 4)
			:SetOnGainInteractFocusFn(OnActivate)
			:SetOnLoseInteractFocusFn(OnDeactivate)
			:SetInteractConditionFn(CanInteract)
			:SetOnInteractFn(PerformInteract)
			:SetLockOnFocus(true)
			:SetCanConditionBreakFocus(false)
			:SetInteractStateName("talk")
			:SetAbortStateName("unsheathe_fast")

		inst:AddComponent("timer")

		inst:AddComponent("conversation")

		inst:AddComponent("npc")
			:SetSpecies(params.species)
		inst.components.npc:SetOnHomeChangedFn(OnHomeChanged)
		inst.components.npc.role = params.role
		inst.components.npc.initial_state = params.initial_state or "VISITOR"

		inst:AddComponent("markablenpc")

		inst:AddComponent("talkaudio")
		inst:AddComponent("foleysounder")
		inst.components.foleysounder:SetFootstepSound(fmodtable.Event.base_layer)
		inst.components.foleysounder:SetSize("medium")

		inst:AddComponent("cineactor")

		inst:SetStateGraph("sg_npc")
		inst:SetBrain("brain_core_npc")

		if params.default_feeling and params.default_feeling ~= "none" then
			inst.default_feeling = params.default_feeling
			inst:PushEvent("feeling", params.default_feeling)
		end

		inst.state = inst.components.npc.initial_state

		inst.OnSave = OnSave
		inst.OnLoad = OnLoad

		if params.role == "specialeventhost" then
			inst:AddComponent("combat")
		end

		TheWorld:PushEvent("registernpc", { npc = inst, role = params.role or Npc.Role.s.visitor })

		prefabutil.ApplyScript(inst, name, params.script, params.script_args)

		return inst
	end

	return Prefab(name, fn, assets, prefabs, nil, NetworkType_Minimal)
end

local ret = {}

for name, params in pairs(NpcAutogenData) do
	ret[#ret + 1] = MakeAutogenNpc(name, params)
	ret[#ret + 1] = MakeAutogenMouth(name, params)
end

-- Don't need group prefabs for npcs
--prefabutil.CreateGroupPrefabs(NpcAutogenData, ret)

return table.unpack(ret)
